<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Check if the table already exists
        if (!Schema::hasTable('students')) {
            Schema::create('students', function (Blueprint $table) {
                $table->id(); // Primary key: Auto-incrementing ID for the student.
                
                // Basic student information
                $table->string('full_name'); // Full name of the student (required).
                $table->string('father_name'); // Father's name (required).
                $table->string('father_job'); // Father's occupation (required).
                $table->string('mother_name'); // Mother's name (required).
                $table->string('mother_job'); // Mother's occupation (required).
                $table->date('date_of_birth'); // Student's date of birth (required).
                $table->enum('religion', ['muslim', 'christian']); // Religion (either Muslim or Christian, required).
                $table->enum('gender', ['male', 'female']); // Gender (male or female, required).

                // Phone number details
                $table->string('home_phone_number')->nullable(); // Home phone number (optional).
                $table->boolean('is_transferred')->default(false); // Indicates if the student is transferred from another school (default is false).
                $table->string('former_school_name')->nullable(); // Former school name (if transferred, required; otherwise, optional).

                // Address and location details
                $table->text('address'); // Detailed address (required).
                $table->string('nationality'); // Student's nationality (required).
                $table->string('place_of_birth'); // Place of birth (required).
                $table->string('national_id')->unique(); // National ID (unique, required).
                $table->year('year_of_entry'); // Year the student joined the school (required).

                // File uploads
                $table->string('photo_path')->nullable(); // Path for the student's photo (optional).
                $table->string('birth_certificate_path')->nullable(); // Path for the student's birth certificate (optional).

                // Activation status
                $table->boolean('is_active')->default(true); // Indicates if the student is currently active (default is true).

                // Foreign keys and relationships
                $table->unsignedBigInteger('grade_id'); // Foreign key for the grade the student is assigned to.
                $table->unsignedBigInteger('school_id'); // Foreign key for the school the student belongs to.
                $table->unsignedBigInteger('neighborhood_id'); // Foreign key for the neighborhood where the student lives.
                $table->unsignedBigInteger('created_by'); // Foreign key for the user (employee) who created the student record.

                // Self-relation for siblings
                $table->unsignedBigInteger('sibling_id')->nullable(); // Self-referencing foreign key for siblings (optional).

                $table->timestamps(); // Timestamps for created_at and updated_at.
                $table->softDeletes(); // Soft delete column for soft deleting the student record.

                // Foreign key constraints
                $table->foreign('grade_id')->references('id')->on('grades')->onDelete('restrict'); // Prevent deletion if the grade has students.
                $table->foreign('school_id')->references('id')->on('schools')->onDelete('restrict'); // Prevent deletion if the school has students.
                $table->foreign('neighborhood_id')->references('id')->on('neighborhoods')->onDelete('restrict'); // Prevent deletion if the neighborhood has students.
                $table->foreign('created_by')->references('id')->on('users')->onDelete('cascade'); // Cascade deletion if the user (employee) is deleted.
                $table->foreign('sibling_id')->references('id')->on('students')->onDelete('cascade'); // Cascade deletion for sibling relationships.
            });
        }
    }

    /**
     * Reverse the migrations.
     *
     * This method drops the foreign key constraints and then drops the students table,
     * effectively undoing the up() method.
     *
     * @return void
     */
    public function down()
    {
        Schema::table('students', function (Blueprint $table) {
            // Drop foreign key constraints before dropping the table
            $table->dropForeign(['grade_id']);
            $table->dropForeign(['school_id']);
            $table->dropForeign(['neighborhood_id']);
            $table->dropForeign(['created_by']);
            $table->dropForeign(['sibling_id']);
        });

        // Now safely drop the students table
        Schema::dropIfExists('students');
    }
};