<?php

namespace App\Http\Middleware\App;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use App\Models\User;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Artisan;

class EnsureSystemIntegrity
{
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        // Check if the current request is for the installation route
        if ($request->is('install') || $request->is('install/*')) {
            // Bypass the middleware for installation routes
            return $next($request);
        }

        // Step 1: Check if the database connection is possible
        try {
            DB::connection()->getPdo();
            Log::info('Database connection established successfully.');
        } catch (\Exception $e) {
            Log::error('Database connection failed: ' . $e->getMessage());
            // Redirect to the installation page if the database connection fails
            return redirect('/install')->with('error', 'Database connection not established. Please set up the database.');
        }

        // Step 2: Check if there are any tables in the database
        $tables = DB::select('SHOW TABLES');
        Log::info('Tables found: ' . json_encode($tables));

        if (empty($tables)) {
            // If no tables exist, redirect to the installation page
            return redirect('/install')->with('error', 'The database is empty. Please complete the installation process.');
        }

        // Step 3: Check if the migrations table exists (to verify if migrations have been run)
        if (!Schema::hasTable('migrations')) {
            Log::warning('Migrations table does not exist.');
            return redirect('/install')->with('error', 'The database migrations have not been run. Please complete the installation process.');
        }

        Log::info('Migrations table exists.');

        // Step 4: Check if the super admin role exists
        $superAdminRole = Role::where('name', 'Super Admin')->first();

        if (!$superAdminRole) {
            // If the Super Admin role does not exist, run the SuperAdminRoleSeeder
            Log::info('Super Admin role not found. Running SuperAdminRoleSeeder.');
            Artisan::call('db:seed', ['--class' => 'Database\\Seeders\\SuperAdminRoleSeeder']);
            $superAdminRole = Role::where('name', 'Super Admin')->first();
        }

        // Step 5: Check if the super admin user exists
        $superAdminUser = User::find(1); // Fetch the super admin user (assuming the ID is 1; adjust as needed)

        if (!$superAdminUser) {
            // If the Super Admin user does not exist, run the AdminSeeder
            Log::info('Super Admin user not found. Running AdminSeeder.');
            Artisan::call('db:seed', ['--class' => 'Database\\Seeders\\AdminSeeder']);
            $superAdminUser = User::find(1); // Fetch the user again after seeding
        }

        // Step 6: Check if the super admin exists
        if (!$superAdminUser || !$superAdminUser->hasRole('Super Admin')) {
            if ($superAdminRole && $superAdminUser) {
                // Assign the role if the super admin user exists but lacks the role
                $superAdminUser->assignRole('Super Admin');
            } else {
                // If the super admin user is missing, redirect to a specific page or an error page
                Log::error('Super Admin account is missing.');
                return redirect('/install')->with('error', 'Super Admin account is missing. Please set up the Super Admin.');
            }
        }

        Log::info('Super Admin verified.');

        // If everything is set up correctly, proceed to the next request
        return $next($request);
    }
}
