<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use App\Models\School;
use App\Models\User;
use App\Models\Log;
use Sentry;

class UserController extends Controller
{
    // Show all users
    public function index()
    {
        $users = User::all();  // Fetch all users
        
        return view('dashboard.users.index', compact('users'));
    }

    // Show form for creating a new user
    public function create()
    {
        $schools = School::all();  // Get all schools to populate the dropdown
        $user = new User; // Pass empty User object
        $roles = Role::all();  // Fetch all available roles

        return view('dashboard.users.form')->with(compact('user', 'roles', 'schools'));
    }

    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users',
            'phone' => 'nullable|string',
            'address' => 'nullable|string',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'roles' => 'required|array',
            'school_id' => 'required|exists:schools,id', // Ensure the school exists
            'hired_date' => 'nullable|date',
        ]);
    
        // Handle file upload
        if ($request->hasFile('photo')) {
            $fileName = time() . '.' . $request->photo->extension();
            $request->photo->move(public_path('images'), $fileName);
            $validatedData['photo'] = $fileName;
        }
        
        // Ensure Super Admin role is included if necessary
        if (in_array($user->id, config('constants.users.SUPER_ADMIN_IDS'))) {
            $superAdminRole = Role::where('name', 'Super Admin')->first();
            
            if (!$superAdminRole) {
                return redirect()->back()->with('error', 'Super Admin role not found.');
            }

            $user->syncRoles(array_unique(array_merge($validatedData['roles'], [$superAdminRole->id])));
        } else {
            $user->syncRoles($validatedData['roles']);
        }

        User::create($validatedData);
    
        return redirect()->route('users.index')->with('success', 'User created successfully.');
    }

    /**
     * Display single user details including related children.
     */
    public function show(User $user)
    {
        // Find the user with its related children data
        $user->load(['school']);

        // Pass the school data to the view
        return view('dashboard.users.show', compact('user'));
    }


    // Show form for assigning roles to a specific user
    public function edit(User $user)
    {
        $schools = School::all();  // Get all schools for dropdown
        $roles = Role::all();  // Fetch all available roles
        $userRoles = $user->roles->pluck('id')->toArray();  // Get roles assigned to the user
        $superAdminRole = Role::where('name', 'Super Admin')->first(); // Define the Super Admin role (make sure this role exists in your database)

        return view('dashboard.users.form', compact('user', 'roles', 'userRoles', 'schools', 'superAdminRole'));
    }

    // Handle role assignment to the user
    public function update(Request $request, User $user)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'nullable|string',
            'address' => 'nullable|string',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'roles' => 'required|array',  // Validate that roles are selected
            'school_id' => 'required|exists:schools,id', // Ensure the school exists
            'hired_date' => 'nullable|date',
        ]);

        // Handle file upload
        if ($request->hasFile('photo')) {
            // Delete the old photo if exists
            if ($user->photo && file_exists(public_path('images/' . $user->photo))) {
                unlink(public_path('images/' . $user->photo));
            }
            $fileName = time() . '.' . $request->photo->extension();
            $request->photo->move(public_path('images'), $fileName);
            $validatedData['photo'] = $fileName;
        }

        // Sync the selected roles with the user
        $user->syncRoles($request->roles);

        // Update user information
        $user->update($validatedData);

        // Log the user update action
        Log::create([
            'user_id' => Auth::id(),
            'action' => 'Updated user',
            'details' => 'Updated user ID ' . $user->id,
            'ip_address' => $request->ip(),
            'user_agent' => $request->header('User-Agent'),
        ]);

        return redirect()->route('users.index')->with('success', 'Roles assigned successfully to the user');
    }

    // Delete a user
    public function destroy(User $user)
    {
        // Assuming 'Super Admin' is the name of the role
        if ($user->hasRole('Super Admin')) {
            return redirect()->route('users.index')->with('error', 'Super Admin cannot be deleted.');
        }

        if ($user->photo && file_exists(public_path('images/' . $user->photo))) {
            unlink(public_path('images/' . $user->photo));
        }

        $user->delete();

        // Log the user update action
        Log::create([
            'user_id' => Auth::id(),
            'action' => 'Deleted user',
            'details' => 'Deleted user ID ' . $user->id,
            'ip_address' => $request->ip(),
            'user_agent' => $request->header('User-Agent'),
        ]);

        // Another log action record using Sentry service
        Sentry::captureMessage('User '.$user->name.' was deleted');

        return redirect()->route('users.index')->with('success', 'User deleted successfully.');
    }

    public function updateRoles(Request $request, User $user)
    {
        $validatedData = $request->validate([
            'roles' => 'required|array',
        ]);
    
        // Check if the user is a Super Admin and ensure the role is not removed
        if (in_array($user->id, config('constants.users.SUPER_ADMIN_IDS'))) {
            // Always make sure the Super Admin role is included
            $superAdminRole = Role::where('name', 'Super Admin')->first();
            if (!$superAdminRole) {
                return redirect()->back()->with('error', 'Super Admin role not found.');
            }
    
            // Ensure the Super Admin role is not removed
            if (!in_array($superAdminRole->id, $validatedData['roles'])) {
                $validatedData['roles'][] = $superAdminRole->id;
            }
        }
    
        // Update the roles
        $user->syncRoles($validatedData['roles']);

        return redirect()->route('users.index')->with('success', 'User roles updated successfully.');
    }

    public function activate(User $user)
    {
        $user->update(['active' => true]);

        // Send activation notification
        $user->notify(new \App\Notifications\UserActivated());

        return redirect()->back()->with('success', 'User has been activated.');
    }

    public function deactivate(User $user)
    {
        $user->update(['active' => false]);

        // Send deactivation notification
        $user->notify(new \App\Notifications\UserDeactivated());

        return redirect()->back()->with('success', 'User has been deactivated.');
    }
}