<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use App\Models\Student;
use App\Models\School;
use App\Models\Grade;
use App\Models\Neighborhood;
use App\Models\User;
use App\Models\Contact;

class StudentController extends Controller
{
/**
     * Display a listing of the students.
     */
    public function index()
    {
        // Fetch all students information
        $students = Student::with(['school', 'grade', 'neighborhood', 'user', 'contacts', 'siblings'])->paginate(20);

        // Return the view with all students information view
        return view('dashboard.students.index', compact('students'));
    }

    /**
     * Show the form for creating a new student.
     */
    public function create()
    {
        $schools = School::all();
        $grades = Grade::all();
        $neighborhoods = Neighborhood::all();

        // Return the view with create new student view
        return view('dashboard.students.create', compact('schools', 'grades', 'neighborhoods'));
    }

    /**
     * Store a newly created student in storage.
     */
    public function store(Request $request)
    {
        // Validation rules
        $validated = $request->validate([
            'full_name' => 'required|string',
            'father_name' => 'required|string',
            'father_job' => 'required|string',
            'mother_name' => 'required|string',
            'mother_job' => 'required|string',
            'date_of_birth' => 'required|date',
            'religion' => 'required|in:muslim,christian',
            'gender' => 'required|in:male,female',
            'address' => 'required|string',
            'nationality' => 'required|string',
            'place_of_birth' => 'required|string',
            'birth_date' => 'required|date',
            'national_id' => 'required|string|unique:students',
            'year_of_entry' => 'required|date_format:Y',
            'grade_id' => 'required|exists:grades,id',
            'school_id' => 'required|exists:schools,id',
            'neighborhood_id' => 'required|exists:neighborhoods,id',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'birth_certificate' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'contacts' => 'required|array|min:1',
            'contacts.*.contact_number' => 'required|string',
            'contacts.*.type' => 'required|in:home,mobile',
            'contacts.*.owned_by' => 'required|in:father,mother,other',
            'contacts.*.notes' => 'nullable|string'
        ]);

        // Handle file uploads
        if ($request->hasFile('photo')) {
            $validated['photo_path'] = $request->file('photo')->store('students/photos', 'public');
        }
        if ($request->hasFile('birth_certificate')) {
            $validated['birth_certificate_path'] = $request->file('birth_certificate')->store('students/certificates', 'public');
        }

        // Automatically assign the 'created_by' field
        $validated['created_by'] = Auth::id();


        // Create the student
        $student = Student::create($validated);

        // Attach siblings if provided
        if ($request->has('sibling_ids')) {
            // Explode the input to get the sibling IDs as an array
            $siblingIds = explode(',', $request->input('sibling_ids'));
            
            // Filter out empty or non-numeric sibling IDs
            $validSiblingIds = array_filter($siblingIds, function ($id) {
                return is_numeric($id) && $id > 0;
            });

            // Sync the valid sibling IDs with the student
            $student->siblings()->sync($validSiblingIds);
        }

        // Handle the contacts
        if ($request->has('contacts')) {
            foreach ($request->contacts as $contact) {
                $student->contacts()->create($contact);
            }
        }

        // Save the note if provided
        if ($request->filled('note')) {
            $student->notes()->create([
                'content' => $request->input('note'),
                'user_id' => auth()->id(),  // Save the ID of the logged-in user
            ]);
        }

        return redirect()->route('admin.students.index')->with('success', 'Student created successfully.');
    }

    /**
     * Display the specified student.
     */
    public function show(Student $student)
    {
        // Fetch single student information
        $student->load(['school', 'grade', 'neighborhood', 'user', 'contacts', 'siblings', 'notes']);

        // Return the view with show single student view
        return view('dashboard.students.show', compact('student'));
    }

    /**
     * Show the form for editing the specified student.
     */
    public function edit(Student $student)
    {
        $schools = School::all();
        $grades = Grade::all();
        $neighborhoods = Neighborhood::all();
        
        // Fetch single student information
        $student->load(['contacts', 'siblings', 'notes']);

        // Return the view with edit single student view
        return view('dashboard.students.edit', compact('student', 'schools', 'grades', 'neighborhoods'));
    }

    /**
     * Update the specified student in storage.
     */
    public function update(Request $request, Student $student)
    {
        // Validation rules
        $validated = $request->validate([
            'full_name' => 'required|string',
            'father_name' => 'required|string',
            'father_job' => 'required|string',
            'mother_name' => 'required|string',
            'mother_job' => 'required|string',
            'date_of_birth' => 'required|date',
            'religion' => 'required|in:muslim,christian',
            'gender' => 'required|in:male,female',
            'address' => 'required|string',
            'nationality' => 'required|string',
            'place_of_birth' => 'required|string',
            'birth_date' => 'required|date',
            'national_id' => 'required|string|unique:students,national_id,' . $student->id,
            'year_of_entry' => 'required|date_format:Y',
            'grade_id' => 'required|exists:grades,id',
            'school_id' => 'required|exists:schools,id',
            'neighborhood_id' => 'required|exists:neighborhoods,id',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'birth_certificate' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'contacts' => 'required|array|min:1',
            'contacts.*.contact_number' => 'required|string',
            'contacts.*.type' => 'required|in:home,mobile',
            'contacts.*.owned_by' => 'required|in:father,mother,other',
            'contacts.*.notes' => 'nullable|string'
        ]);

        // Handle file uploads
        if ($request->hasFile('photo')) {
            // Delete the old photo if exists
            if ($student->photo_path) {
                Storage::disk('public')->delete($student->photo_path);
            }
            $validated['photo_path'] = $request->file('photo')->store('students/photos', 'public');
        }

        if ($request->hasFile('birth_certificate')) {
            // Delete the old birth certificate if exists
            if ($student->birth_certificate_path) {
                Storage::disk('public')->delete($student->birth_certificate_path);
            }
            $validated['birth_certificate_path'] = $request->file('birth_certificate')->store('students/certificates', 'public');
        }

        // Update student details
        $student->update($validated);

        // Save the selected siblings, ensuring you exclude trashed siblings
        if ($request->has('sibling_ids')) {
            // Explode the input to get the sibling IDs as an array
            $siblingIds = explode(',', $request->input('sibling_ids'));
            
            // Filter out empty or non-numeric sibling IDs
            $validSiblingIds = array_filter($siblingIds, function ($id) {
                return is_numeric($id) && $id > 0;
            });

            // Sync the valid siblings but exclude trashed ones
            $student->siblings()->whereNull('deleted_at')->sync($validSiblingIds);
        }

        // Update or create contacts
        if ($request->has('contacts')) {
            $student->contacts()->delete(); // Delete existing contacts before adding new ones

            foreach ($request->contacts as $contact) {
                $student->contacts()->create($contact);
            }
        }

        // Save the new note if provided
        if ($request->filled('note')) {
            $student->notes()->create([
                'content' => $request->input('note'),
                'user_id' => auth()->id(),  // Save the ID of the logged-in user
            ]);
        }

        return redirect()->route('students.index')->with('success', 'Student updated successfully.');
    }

    /**
     * Remove the specified student from storage (soft delete).
     */
    public function destroy(Student $student)
    {
        // Soft delete the student
        $student->delete();

        return redirect()->route('students.index')->with('success', 'Student deleted successfully.');
    }

/**
     * Restore a soft-deleted student.
     */
    public function restore($id)
    {
        $student = Student::withTrashed()->findOrFail($id);

        $student->restore();

        return redirect()->route('students.index')->with('success', 'Student restored successfully.');
    }

    /**
     * Permanently delete the specified student from storage.
     */
    public function forceDelete($id)
    {
        $student = Student::withTrashed()->findOrFail($id);

        // Check for relations
        $relatedRecords = [];

        if ($student->contacts()->exists()) {
            $relatedRecords[] = 'contacts';
        }

        if (!empty($relatedRecords)) {
            return back()->withErrors([
                'error' => 'Cannot delete this student because they have related ' . implode(', ', $relatedRecords),
            ]);
        }

        // Delete student photos and birth certificates if they exist
        if ($student->photo_path) {
            Storage::disk('public')->delete($student->photo_path);
        }

        if ($student->birth_certificate_path) {
            Storage::disk('public')->delete($student->birth_certificate_path);
        }

        // Proceed with deletion if no related records and permanently delete the student
        $student->forceDelete();

        return redirect()->route('students.index')->with('success', 'Student permanently deleted.');
    }

    /**
     * Display a listing of soft-deleted students (for admin or management).
     */
    public function trashed()
    {
        // Fetch trashed students
        $students = Student::onlyTrashed()->with(['school', 'grade', 'neighborhood', 'user', 'contacts', 'siblings'])->paginate(20);

        // Return the view with trashed students
        return view('dashboard.students.trashed', compact('students'));
    }

    public function searchSiblings(Request $request)
    {
        $query = $request->get('query');
        
        // Fetch matching students excluding trashed ones
        $students = Student::where('name', 'LIKE', "%{$query}%")
                            ->whereNull('deleted_at') // Make sure trashed students are excluded
                            ->get(['id', 'name']);
        
        // Return the response in JSON format
        return response()->json($students);
    }
}