<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\School;

class SchoolController extends Controller
{
/**
     * Display a listing of the schools, including soft-deleted ones.
     */
    public function index()
    {
        // Retrieve all schools, including the soft-deleted ones
        $schools = School::withTrashed()->paginate(10);

        return view('dashboard.schools.index', compact('schools'));
    }

    /**
     * Show the form for creating a new school.
     */
    public function create()
    {
        return view('dashboard.schools.create');
    }

    /**
     * Store a newly created school in the database.
     */
    public function store(Request $request)
    {
        // Validate the request
        $request->validate([
            'name' => 'required|string|max:255',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'registration_code' => 'required|string|unique:schools|max:255',
            'email' => 'required|string|email|max:255|unique:schools',
            'address' => 'nullable|string',
            'city' => 'required|string|max:255',
            'state' => 'required|string|max:255',
            'country' => 'required|string|max:255',
            'zipcode' => 'nullable|string|max:20',
            'phone' => 'nullable|string|max:15',
            'website' => 'nullable|string|url|max:255',
            'principal_name' => 'nullable|string|max:255',
            'founded_year' => 'nullable|integer|min:1800|max:' . date('Y'),
            'student_capacity' => 'nullable|integer',
            'type' => 'nullable|in:public,private,charter',
            'status' => 'required|in:active,inactive,pending',
        ]);

        $data = $request->all();

        // Handle file upload for the logo
        if ($request->hasFile('logo')) {
            $filePath = $request->file('logo')->store('logos', 'public');
            $data['logo'] = $filePath;  // Save the path to the database
        }

        // Create a new school record
        School::create($request->all());

        return redirect()->route('schools.index')->with('success', 'School created successfully.');
    }

    /**
     * Display single school details including related children.
     */
    public function show(School $school)
    {
        // Find the school with its related children data
        $school->load(['students', 'users']);

        // Pass the school data to the view
        return view('dashboard.schools.show', compact('school'));
    }

    /**
     * Show the form for editing the specified school.
     */
    public function edit(School $school)
    {
        return view('dashboard.schools.edit', compact('school'));
    }

    /**
     * Update the specified school in the database.
     */
    public function update(Request $request, School $school)
    {
        // Validate the request
        $request->validate([
            'name' => 'required|string|max:255',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048', // Validate logo
            'registration_code' => 'required|string|max:255|unique:schools,registration_code,' . $school->id,
            'email' => 'required|string|email|max:255|unique:schools,email,' . $school->id,
            'address' => 'nullable|string',
            'city' => 'required|string|max:255',
            'state' => 'required|string|max:255',
            'country' => 'required|string|max:255',
            'zipcode' => 'nullable|string|max:20',
            'phone' => 'nullable|string|max:15',
            'website' => 'nullable|string|url|max:255',
            'principal_name' => 'nullable|string|max:255',
            'founded_year' => 'nullable|integer|min:1800|max:' . date('Y'),
            'student_capacity' => 'nullable|integer',
            'type' => 'nullable|in:public,private,charter',
            'status' => 'required|in:active,inactive,pending',
        ]);

        $data = $request->all();

        // Handle file upload for the logo
        if ($request->hasFile('logo')) {
            // Delete the old logo if it exists
            if ($school->logo && \Storage::disk('public')->exists($school->logo)) {
                \Storage::disk('public')->delete($school->logo);
            }
    
            $filePath = $request->file('logo')->store('logos', 'public');
            $data['logo'] = $filePath;  // Save the new logo path
        }
        
        // Update the school's information
        $school->update($request->all());

        return redirect()->route('schools.index')->with('success', 'School updated successfully.');
    }

    /**
     * Soft delete the specified school.
     */
    public function destroy(School $school)
    {
        // Soft delete the school
        $school->delete();

        return redirect()->route('schools.index')->with('success', 'School has been soft deleted.');
    }

    /**
     * Restore a soft-deleted school.
     */
    public function restore($id)
    {
        // Find the soft-deleted school
        $school = School::withTrashed()->findOrFail($id);

        // Restore the school
        $school->restore();

        return redirect()->route('schools.index')->with('success', 'School has been restored.');
    }

    /**
     * Permanently delete a soft-deleted school.
     */
    public function forceDelete($id)
    {
        // Find the soft-deleted school
        $school = School::withTrashed()->findOrFail($id);

        // Check if school has any related students or employees
        if ($school->students->count() > 0 || $school->employees->count() > 0) {
            $errorMsg = 'The school cannot be deleted as it has related data:';
            if ($school->students->count() > 0) {
                $errorMsg .= ' Students (' . $school->students->count() . ')';
            }
            if ($school->employees->count() > 0) {
                $errorMsg .= ' Employees (' . $school->employees->count() . ')';
            }

            return back()->withErrors(['error' => $errorMsg]);
        }

        // If no related data, permanently delete the school
        $school->forceDelete();

        return redirect()->route('schools.index')->with('success', 'School has been permanently deleted.');
    }
}