<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Illuminate\Http\Request;

class RoleController extends Controller
{
    // List all roles
    public function index()
    {
        // Retrieve roles with the count of associated permissions
        $roles = Role::withCount('permissions')->get();

        return view('dashboard.roles.index', compact('roles'));
    }

    // Show role and its permissions
    public function show($id)
    {
        $role = Role::with('permissions')->findOrFail($id);
        return view('dashboard.roles.show', compact('role'));
    }

    // Show form for creating a new role with permission selection
    public function create()
    {
        $permissions = Permission::all(); // Fetch all available permissions

        return view('dashboard.roles.create', compact('permissions'));
    }

    // Store a newly created role with assigned permissions
    public function store(Request $request)
    {
        // Validate the request data
        $validatedData = $request->validate([
            'name' => 'required|string|max:255|unique:roles,name',
            'permissions' => 'nullable|array', // 'permissions' expected to be an array of IDs
        ]);

        // Create a new role
        $role = Role::create(['name' => $validatedData['name']]);

        // Check if the role is not "Super Admin"
        if ($role->name != 'Super Admin') {
            // If permissions are provided, fetch them from the database
            if (!empty($validatedData['permissions'])) {
                // Get the permissions by ID
                $permissions = Permission::whereIn('id', $validatedData['permissions'])->get();
                // Sync the valid permissions with the role
                $role->syncPermissions($permissions);
            } else {
                // If no permissions were provided, just sync an empty array
                $role->syncPermissions([]);
            }
        } 
        else {
            // Optionally, you can assign all permissions to Super Admin
            $allPermissions = Permission::all();
            $role->syncPermissions($allPermissions);
        }

        // Redirect back to roles index with success message
        return redirect()->route('roles.index')->with('success', 'Role created and permissions assigned successfully');
    }

    // Show form for editing a role with permission selection
    public function edit(Role $role)
    {
        $permissions = Permission::all(); // Fetch all permissions
        $rolePermissions = $role->permissions()->pluck('id')->toArray(); // Get assigned permissions for the role


        return view('dashboard.roles.edit', compact('role', 'permissions', 'rolePermissions'));
    }

    // Update the role and its permissions
    public function update(Request $request, Role $role)
    {
        // Prevent updating Super Admin role permissions
        if ($role->name == 'Super Admin') {
            return redirect()->route('roles.index')->with('error', 'Cannot update permissions for Super Admin role.');
        }

        // Validate and update role details
        $validatedData = $request->validate([
            'name' => 'required|unique:roles,name,' . $role->id,
            'permissions' => 'required|array',
        ]);

        // Update the role name
        $role->update(['name' => $validatedData['name']]);

        // Fetch the permissions by ID and retrieve their names
        $permissions = Permission::whereIn('id', $validatedData['permissions'])->pluck('name')->toArray();

        // Sync permissions with the role
        $role->syncPermissions($permissions);

        return redirect()->route('roles.index')->with('success', 'Role and permissions updated successfully');
    }


    // Delete a role
    public function destroy(Role $role)
    {
        try {
            // Check if the role has any associated permissions
            if ($role->permissions()->count() > 0) {
                return redirect()->route('roles.index')->with('error', 'This role has assigned permissions and cannot be deleted.');
            }

            // Attempt to delete the role
            $role->delete();

            return redirect()->route('roles.index')->with('success', 'Role deleted successfully.');
        } catch (\Illuminate\Database\QueryException $e) {
            // Check for the SQLSTATE for the Super Admin deletion error
            if ($e->getCode() == '45000') {
                // Log expection using Sentry
                Sentry::captureException($e->getMessage());
                
                return redirect()->route('roles.index')->with('error', 'Super Admin role cannot be removed.');
            }

            // For other errors, display a generic error message
            return redirect()->route('roles.index')->with('error', 'An error occurred while deleting the role.');
        }
    }
}