<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use App\Models\Invitation;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use App\Mail\InvitationMail;
use App\Notifications\InviteAcceptedNotification;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;

class InvitationController extends Controller
{
    // Show list of invitations
    public function index()
    {
        $invitations = Invitation::with('user')->get(); // This will now work correctly
        
        return view('dashboard.invitations.index', compact('invitations'));
    }

    /**
     * Show the form to send an invitation.
     */
    public function create()
    {
        return view('dashboard.invitations.create');
    }

    // Send a new invitation
    public function store(Request $request)
    {
        // Validate the email
        $request->validate([
            'email' => 'required|email|unique:invitations,email', // Ensure unique invitation email
            'name' => 'nullable|string|max:255', // Optional name field validation
        ]);

        // Create a new invitation in the database
        $invitation = Invitation::create([
            'email' => $request->email,
            'name' => $request->name, // Store the optional name field
            'token' => Str::random(32), // Generate a unique token
            'status' => 'pending',
        ]);

        // Send the invitation email
        Mail::to($invitation->email)->send(new InvitationMail($invitation));

        // Redirect back with a success message
        return back()->with('success', 'Invitation sent successfully.');
    }

    // Handle acceptance of the invitation
    public function acceptInvitation($token)
    {
        $invitation = Invitation::where('token', $token)->where('status', 'pending')->firstOrFail();

        // Check if the user already exists (you could add registration here if user doesn't exist)
        $user = User::where('email', $invitation->email)->first();

        if (!$user) {
            return redirect()->route('register')->with('error', 'No user found, please register first.');
        }

        // Mark invitation as accepted
        $invitation->update([
            'status' => 'accepted',
            'user_id' => $user->id,
            'accepted_at' => now(),
        ]);

        // Notify the admin about the accepted invitation
        Notification::route('mail', config('mail.admin_email'))
            ->notify(new InviteAcceptedNotification($user));

        return redirect()->route('dashboard')->with('success', 'Invitation accepted successfully.');
}
}
