<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Grade;

class GradeController extends Controller
{
    // Display a list of grades
    public function index()
    {
        // Retrieve all grades with the count of students under each
        $grades = Grade::withCount('students')->get();

        return view('dashboard.grades.index', compact('grades'));
    }

    // Show form to create a new grade
    public function create()
    {
        return view('dashboard.grades.create');
    }

    // Store a new grade
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
        ]);

        Grade::create($validated);
        return redirect()->route('grades.index');
    }

    // Show a single grade
    public function show(Grade $grade)
    {
        // Load the grade along with paginated students
        $students = $grade->students()->paginate(10); // Paginate with 10 students per page

        // Pass the grade and paginated students to the view
        return view('dashboard.grades.show', compact('grade', 'students'));
    }

    // Show form to edit a grade
    public function edit(Grade $grade)
    {
        return view('dashboard.grades.edit', compact('grade'));
    }

    // Update a grade
    public function update(Request $request, Grade $grade)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
        ]);

        $grade->update($validated);
        return redirect()->route('grades.index');
    }

    // Delete a grade
    public function destroy(Grade $grade)
    {
        try {
            // Check if the grade has any students before attempting to delete it
            if ($grade->students()->count() > 0) {
                // Redirect back with an error message if there are students
                return redirect()->route('grades.index')->with('error', 'Cannot delete a grade that has students assigned to it.');
            }
        
            // If no students, delete the grade
            $grade->delete();
        
            // Redirect back with a success message
            return redirect()->route('grades.index')->with('success', 'Grade deleted successfully.');
        } 
        catch (QueryException $e) {
            // Handle any SQL error, such as foreign key constraint violation
            return redirect()->route('grades.index')->with('error', 'Error: Cannot delete this grade because it is linked to other records.');
        }
    }
}
