<?php

namespace App\Http\Controllers\Dashboard\Auth;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use App\Models\User;
use App\Models\Profile;
use App\Models\Invitation;
use App\Providers\RouteServiceProvider;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Spatie\Permission\Models\Permission;

class RegisteredUserController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:invitations-list', ['only' => ['index']]);
        $this->middleware('permission:invitations-send-request', ['only' => ['requestInvitation']]);
    }
    
    /**
     * Display the registration view.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        addJavascriptFile('assets/js/custom/authentication/sign-up/general.js');

        return view('pages.auth.register');
    }
    
    /**
     * Handle an incoming registration request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request)
    {      
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
        ]);

        if (User::where('email', '=', $request->email)->exists()) {
            Session::flash('error', 'Sorry but this email already exists.');

            return redirect()->back();
        }
         
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);
        
        if($request->invitation_token) {
            $invitation = Invitation::where('invitation_token', $request->invitation_token)->firstOrFail();
            $invitation->registered_at = date('Y-m-d H:i:s');
            $invitation->receiver_user_id = $user->id;
            
            $invitation->save();
        }
        
        if($request->permissions) {
            $permissions = unserialize(base64_decode($request->permissions));

            $user->syncPermissions($permissions);
        }

        event(new Registered($user));

        //$user->profile()->create(['user_id' => $user->id, 'username' => $username, 'created_at' => date("Y-m-d H:i:s"), 'updated_at' => date("Y-m-d H:i:s")]);
        
        $profile                = new Request();
        $profile->username      = generateUserName($request->email);
        $profile->created_at    = date("Y-m-d H:i:s");
        $profile->updated_at    = date("Y-m-d H:i:s");
   
        generateProfle($profile, $user->id); //Check if user has a profile record in profiles table .. If not then create a record else update record

        Auth::login($user);

        $request->user()->update([
            'last_login_at' => Carbon::now()->toDateTimeString(),
            'last_login_ip' => $request->getClientIp()
        ]);
        
        return redirect(RouteServiceProvider::HOME);
    }

    /**
     * Display the request invitations view.
     *
     * @return \Illuminate\View\View
     */
    public function requestInvitation() { 
        $permission = Permission::get();

        return view('pages.auth.request',compact('permission'));
    }

    /**
     * Display the registration view.
     *
     * @return \Illuminate\View\View
     */
    public function showRegistrationForm(Request $request)
    {
        if (Auth::check()) {
            return redirect(RouteServiceProvider::HOME);
        }

        $invitation_token = $request->get('invitation_token');
        $invitation = Invitation::where('invitation_token', $invitation_token)->firstOrFail();
        $email = $invitation->email;
    
        return view('pages.auth.request-register', compact('email'));
    }
}
