<?php

// app/Helpers/SettingsHelper.php
use App\Models\Setting;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Schema;

if (! function_exists('setting')) {
    /**
     * Get or set a setting value.
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    function setting($key, $default = null)
    {
        // Check if 'settings' table exists
        if (Schema::hasTable('settings')) {
            // Use cache to reduce DB queries
            return Cache::rememberForever('setting_' . $key, function () use ($key, $default) {
                $setting = Setting::where('key', $key)->first();
                return $setting ? $setting->value : $default;
            });
        }
    }

    /**
     * Set a setting value.
     *
     * @param string $key
     * @param mixed $value
     * @param string $type
     * @return bool
     */
    function set_setting($key, $value, $type = 'string')
    {
        $setting = Setting::updateOrCreate(
            ['key' => $key],
            ['value' => $value, 'type' => $type]
        );

        // Update cache
        Cache::forever('setting_' . $key, $value);

        return $setting ? true : false;
    }

    /**
     * Check if required settings are present and non-null.
     *
     * @return bool True if settings are missing or null, False if all required settings are present
     */
    function checkSettings()
    {
        // Define the required settings keys to check
        $requiredSettings = [
            'site_name',
            'site_logo',
            'site_favicon',
            'primary_color',
            'site_theme',
            'users_registration'
        ];

        // Check if the 'settings' table exists
        if (!Schema::hasTable('settings')) {
            return true; // If settings table does not exist, redirect to installation
        }

        // Fetch required settings from the database
        $settings = DB::table('settings')
            ->whereIn('key', $requiredSettings)
            ->pluck('value', 'key');

        // Return true if any required setting is missing or null
        foreach ($requiredSettings as $settingKey) {
            if (!isset($settings[$settingKey]) || is_null($settings[$settingKey])) {
                \Log::info("Setting $settingKey is missing or null");
                return true;
            }
        }

        return false; // All required settings are present and non-null
    }
}