<?php

namespace App\Http\Requests\Auth;

use Illuminate\Auth\Events\Lockout;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use App\Events\UserOnlineStatusChanged;
use App\Providers\RouteServiceProvider;

class LoginRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        // Determine if the input is an email or username
        $login = $this->input('login');
        $field = filter_var($login, FILTER_VALIDATE_EMAIL) ? 'email' : 'username';

        return [
            'login' => [
                'required', 
                $field === 'email' ? 'email' : 'string'
            ],
            'password' => ['required', 'string'],
        ];
    }

    /**
     * Attempt to authenticate the request's credentials.
     *
     * @return void
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function authenticate()
    {
        // Ensure rate limiting is respected
        $this->ensureIsNotRateLimited();
    
        // Get the login input (email or username)
        $login = $this->input('login');
        
        // Determine if login is an email or username
        $field = filter_var($login, FILTER_VALIDATE_EMAIL) ? 'email' : 'username';
    
        // Validation based on email or username
        $credentials = $this->validate([
            'login' => ['required', $field === 'email' ? 'email' : 'string'],
            'password' => ['required'],
        ]);
    
        // Attempt to log in using either email or username with the 'admin' guard
        if (Auth::guard('admin')->attempt([$field => $this->login, 'password' => $this->password])) {
            // Find the logged-in admin user
            $user = Auth::guard('admin')->user();
    
            // Check if user is active
            if (!$user->active) {
                // Logout and redirect back with an error if user is deactivated
                Auth::guard('admin')->logout();
                return back()->withErrors(['login' => 'Your account is deactivated. Please contact support.'])->onlyInput('login');
            }
    
            // Store the admin user session for tracking activity
            session(['was_user_logged_in' => $user]);
    
            // Update the user's online status to 'online'
            $user->update(['online' => 1]);
    
            // Optionally broadcast the user’s online status (can be skipped if not needed)
            try {
                event(new UserOnlineStatusChanged($user));
            } catch (\Exception $e) {
                \Log::error('Broadcasting failed: ' . $e->getMessage());
            }
    
            // Regenerate session to avoid fixation attacks
            $this->session()->regenerate();
    
            // Redirect to the intended URL or default admin dashboard
            return redirect()->intended(RouteServiceProvider::ADMIN_HOME);
        }
    
        // If login fails, return back with error message
        //RateLimiter::hit($this->throttleKey());

        return back()->withErrors([
            'login' => 'The provided credentials do not match our records.',
        ])->onlyInput('login');
    }

    /**
     * Ensure the login request is not rate limited.
     *
     * @return void
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function ensureIsNotRateLimited()
    {
        if (! RateLimiter::tooManyAttempts($this->throttleKey(), 5)) {
            return;
        }

        event(new Lockout($this));

        $seconds = RateLimiter::availableIn($this->throttleKey());

        throw ValidationException::withMessages([
            'email' => trans('auth.throttle', [
                'seconds' => $seconds,
                'minutes' => ceil($seconds / 60),
            ]),
        ]);
    }

    /**
     * Get the rate limiting throttle key for the request.
     *
     * @return string
     */
    public function throttleKey()
    {
        return Str::transliterate(Str::lower($this->input('email')).'|'.$this->ip());
    }
}
