<?php

namespace App\Http\Controllers\System;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use App\Models\Setting;

class InstallationController extends Controller
{
    // Display the installation page
    public function index()
    {
        try {
            // Check if we can connect to the database
            DB::connection()->getPdo();
            
            // Get all tables in the connected database
            $tables = DB::select('SHOW TABLES');

            if (empty($tables)) {
                // If no tables exist, show a message and a "Build Database" button
                return view('install.build-database');
            } else {
                // If tables already exist, show a message and don't show the form
                return redirect()->route('install.complete')->with('success', 'Database built successfully!');
            }
        } catch (\Exception $e) {
            // If no DB connection, show the installation form
            return view('install.index', ['tablesExist' => false]);
        }
    }

    // Update Installation Step to Handle Dynamic Setting Creation
    public function initializeSettings()
    {
        $requiredSettings = [
            'site_name' => 'text',
            'site_logo' => 'file',
            'site_favicon' => 'file',
            'primary_color' => 'color',
            'site_theme' => 'select',
            'users_registration' => 'boolean',
        ];
    
        foreach ($requiredSettings as $key => $type) {
            if($key == 'site_theme') {
                Setting::firstOrCreate(
                    ['key' => $key],
                    ['value' => json_encode(['light', 'dark', 'blue']), 'type' => $type, 'group' => 'general']
                );
            }
            else {
                Setting::firstOrCreate(
                    ['key' => $key],
                    ['value' => null, 'type' => $type, 'group' => 'general']
                );
            }
        }
    }

    // Setup the database and run migrations/seeds
    public function setup(Request $request)
    {
        // Save the database credentials to the .env file
        $envContent = File::get(base_path('.env'));

        // Replace or add the database credentials
        $envContent = preg_replace('/DB_HOST=.*/', 'DB_HOST=' . $request->input('DB_HOST'), $envContent);
        $envContent = preg_replace('/DB_PORT=.*/', 'DB_PORT=' . $request->input('DB_PORT'), $envContent);
        $envContent = preg_replace('/DB_DATABASE=.*/', 'DB_DATABASE=' . $request->input('DB_DATABASE'), $envContent);
        $envContent = preg_replace('/DB_USERNAME=.*/', 'DB_USERNAME=' . $request->input('DB_USERNAME'), $envContent);
        $envContent = preg_replace('/DB_PASSWORD=.*/', 'DB_PASSWORD=' . $request->input('DB_PASSWORD'), $envContent);

        File::put(base_path('.env'), $envContent);

        // Clear Laravel caches
        try {
            Artisan::call('config:clear');
            Artisan::call('cache:clear');
            Artisan::call('route:clear');
            Artisan::call('view:clear');
            Artisan::call('optimize:clear');

            // Run migrations and seeders
            Artisan::call('migrate');
            Artisan::call('db:seed');
            
            // Initialize settings if not already present
            $this->initializeSettings();
    
            return redirect()->route('home')->with('success', 'Installation completed successfully!');
        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => $e->getMessage()]);
        }
    }

    // Run migrations and seed the database
    public function buildDatabase()
    {
        try {
            // Automatically clear and optimize Laravel caches
            Artisan::call('config:clear');
            Artisan::call('cache:clear');
            Artisan::call('route:clear');
            Artisan::call('view:clear');
            Artisan::call('optimize:clear');

            // Run migrations and seeders
            Artisan::call('migrate');
            Artisan::call('db:seed');
            
            // Initialize settings if not already present
            $this->initializeSettings();

            return redirect()->route('install.complete')->with('success', 'Database built successfully!');
        } catch (\Exception $e) {
            return redirect()->route('install.index')->withErrors(['error' => $e->getMessage()]);
        }
    }
    
    public function migrateAndRedirectToSettings()
    {
        // Run migrations
        Artisan::call('migrate');
    
        // Check if the settings table is empty
        if (Setting::count() === 0) {
            // Redirect to settings input form
            return redirect()->route('install.settings');
        }

        return redirect()->route('front.pages.home');
    }
    
    public function storeSettings(Request $request)
    {
        $request->validate([
            'site_name' => 'required|string',
            'site_logo' => 'nullable|image',
            'site_favicon' => 'nullable|image',
            'primary_color' => 'required|string',
            'site_theme' => 'required|in:light,dark',
            'users_registration' => 'required|boolean',
            // Additional validation rules if needed
        ]);
    
        // Store settings in the database
        Setting::updateOrCreate(['key' => 'system_name'], ['value' => $request->input('site_name')]);
        Setting::updateOrCreate(['key' => 'primary_color'], ['value' => $request->input('primary_color')]);
        Setting::updateOrCreate(['key' => 'site_theme'], ['value' => $request->input('site_theme')]);
        Setting::updateOrCreate(['key' => 'users_registration'], ['value' => $request->input('users_registration')]);

        // Handle file uploads for site_logo and site_favicon
        if ($request->hasFile('site_logo')) {
            $logoPath = $request->file('site_logo')->store('settings', 'public');
            Setting::updateOrCreate(['key' => 'site_logo'], ['value' => $logoPath]);
        }
        if ($request->hasFile('site_favicon')) {
            $faviconPath = $request->file('site_favicon')->store('settings', 'public');
            Setting::updateOrCreate(['key' => 'site_favicon'], ['value' => $faviconPath]);
        }

        // After saving settings, allow access to the rest of the system
        return redirect()->route('install.complete')->with('success', 'Installation completed successfully!');
    }

    // Show form to show default settings page
    public function getDefaultSettings()
    {
        return view('install.settings');
    }

    // Store default settings
    public function storeDefaultSettings(Request $request)
    {
        try {
            // Define validation rules
            $request->validate([
                'site_name' => 'required|string|max:255',
                'site_logo' => 'required|file|mimes:jpeg,png,jpg,gif|max:2048',
                'site_favicon' => 'required|file|mimes:jpeg,png,jpg,gif,ico|max:2048',
                'primary_color' => 'required|string',
                'site_theme' => 'required|in:light,dark',
                'users_registration' => 'required|boolean',
            ], [
                'site_name.required' => 'The site name is required.',
                'site_name.string' => 'The site name must be a valid string.',
                'site_name.max' => 'The site name may not be greater than 255 characters.',
            
                'site_logo.required' => 'Please upload a site logo.',
                'site_logo.file' => 'The site logo must be a valid file.',
                'site_logo.mimes' => 'The site logo must be a file of type: jpeg, png, jpg, gif.',
                'site_logo.max' => 'The site logo may not be larger than 2MB.',
            
                'site_favicon.required' => 'Please upload a site favicon.',
                'site_favicon.file' => 'The site favicon must be a valid file.',
                'site_favicon.mimes' => 'The site favicon must be a file of type: jpeg, png, jpg, gif, ico.',
                'site_favicon.max' => 'The site favicon may not be larger than 2MB.',
            
                'primary_color.required' => 'Please select a primary color for the site.',
                'primary_color.string' => 'The primary color must be a valid color code.',
            
                'site_theme.required' => 'Please select a site theme.',
                'site_theme.in' => 'The selected site theme is invalid. Please choose either light or dark.',
            
                'users_registration.required' => 'Please select whether to allow user registration.',
                'users_registration.boolean' => 'The users registration field must be either "Allow" or "Deny".'
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            \Log::error('Validation failed:', $e->errors()); // Log errors for debugging
            return back()->withErrors($e->errors())->withInput();
        }

        // Define the settings fields
        $fields = [
            'site_name' => [
                'type' => 'text',
                'value' => $request->input('site_name')
            ],
            'site_logo' => [
                'type' => 'file',
                'value' => $request->file('site_logo')->store('settings', 'public')
            ],
            'site_favicon' => [
                'type' => 'file',
                'value' => $request->file('site_favicon')->store('settings', 'public')
            ],
            'primary_color' => [
                'type' => 'color',
                'value' => $request->input('primary_color')
            ],
            'site_theme' => [
                'type' => 'select',
                'value' => $request->input('site_theme')
            ],
            'users_registration' => [
                'type' => 'radio',
                'value' => $request->input('users_registration')
            ],
        ];
    
        // Store or update each setting in the database
        foreach ($fields as $key => $data) {
            Setting::updateOrCreate(
                ['key' => $key],
                [
                    'value' => $data['value'],
                    'type' => $data['type'],
                    'group' => 'general',
                ]
            );
        }
    
        // After saving settings get redirected to complete page
        return redirect()->route('install.complete')->with('success', 'Settings saved successfully!');
    }
    
    public function complete()
    {
        // Check if the migrations table exists (to verify if migrations have been run)
        if (!Schema::hasTable('migrations')) {
            return redirect('/install')->with('error', 'The database is not set up. Please complete the installation process.');
        }
        // Check if the required settings are present in the database
        elseif (checkSettings()) {
            return redirect('/install/settings')->with('error', 'Some required settings are missing. Please complete the settings installation process.');
        }
        else {
            // Show the installation complete page after the database has been built
            return view('install.complete');
        }
    }
}