<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Setting;
use Illuminate\Support\Facades\Storage;

class SettingController extends Controller
{
    // Display settings grouped by their categories
    public function index($group = null)
    {       
        // Retrieve the settings for the specified group
        $settings = Setting::where('group', $group)->get()->map(function($setting) {
            // Decode value if it's a JSON string and the setting is of type 'select'
            if ($setting->type === 'select' && $this->isJson($setting->value)) {
                $setting->options = json_decode($setting->value, true);
            } else {
                // Otherwise, use the raw value
                $setting->options = null;
            }
            return $setting;
        });

        return view('dashboard.settings.index', compact('settings', 'group'));
    }

    // Show form to create new setting
    public function create()
    {
        // Show form to create a new setting
        return view('dashboard.settings.create');
    }

    // Store the new setting in the database
    public function store(Request $request)
    {
        // Validate input based on type
        $request->validate([
            'key' => 'required|string|unique:settings',
            'type' => 'required|string',
            'group' => 'required|string',
            'value' => $this->getValidationRule($request->type),
            'options' => 'nullable|array' // To handle options for dropdown and radio buttons
        ]);

        // Convert options to JSON if they exist
        $options = $request->has('options') ? json_encode($request->options) : null;

        // Handle file upload if type is file
        $value = $request->value;
        if ($request->hasFile('value')) {
            $file = $request->file('value');
            $value = $file->store('uploads', 'public'); // Save file to storage
        }

        // Create the setting in the database
        Setting::create([
            'key' => $request->key,
            'value' => $value,
            'type' => $request->type,
            'group' => $request->group,
            'options' => $options,
        ]);

        return redirect()->route('settings.index')->with('success', 'Setting created successfully.');
    }

    // Edit a specific setting
    public function edit($id)
    {
        $setting = Setting::findOrFail($id);
        return view('settings.edit', compact('setting'));
    }

    // Update an existing setting
    public function update(Request $request)
    {
        $settings = Setting::all();
    
        foreach ($settings as $setting) {
            $key = $setting->key;
            $type = $setting->type;
            $value = $request->input($key);
    
            // Validation based on type (add this function if not yet defined)
            $request->validate([
                $key => $this->getValidationRule($type),
            ]);
    
            switch ($type) {
                case 'text':
                case 'textarea':
                case 'date':
                case 'color':
                case 'radio':
                case 'select':
                    // Directly assign the input value
                    $setting->value = $value;
                    break;
    
                case 'checkbox':
                case 'boolean':
                    // Set value to '1' if checked, '0' if unchecked
                    $setting->value = $request->has($key) ? '1' : '0';
                    break;
    
                case 'json':
                    // Handle multi-select or array-based fields as JSON
                    if (is_array($value)) {
                        $setting->value = json_encode($value);
                    }
                    break;
    
                case 'file':
                    // Handle single file upload
                    if ($request->hasFile($key)) {
                        // Delete old file if it exists
                        if ($setting->value) {
                            Storage::disk('public')->delete($setting->value);
                        }
                        // Store new file in the "settings" directory
                        $filePath = $request->file($key)->store('settings', 'public');
                        $setting->value = $filePath;
                    }
                    break;
    
                case 'file_multiple':
                    // Handle multiple file uploads (stored as JSON)
                    if ($request->hasFile($key)) {
                        // Delete old files if necessary
                        if ($setting->value) {
                            $oldFiles = json_decode($setting->value, true);
                            foreach ($oldFiles as $oldFile) {
                                Storage::disk('public')->delete($oldFile);
                            }
                        }
    
                        // Store new files in the "settings" directory
                        $files = [];
                        foreach ($request->file($key) as $file) {
                            $files[] = $file->store('settings', 'public');
                        }
                        $setting->value = json_encode($files);
                    }
                    break;
                
                default:
                    break;
            }
    
            // Save each updated setting
            $setting->save();
        }
    
        return redirect()->back()->with('success', 'Settings updated successfully!');
    }

    // Return validation rule based on setting type
    private function getValidationRule($type)
    {
        switch ($type) {
            case 'file':
                return 'file|mimes:jpg,jpeg,png,gif,pdf,docx|max:2048'; // File validation
            case 'boolean':
                return 'boolean';
            case 'array':
                return 'array';
            case 'string':
            default:
                return 'nullable|string';
        }
    }

    // Delete a setting from the database
    public function destroy($id)
    {
        $setting = Setting::findOrFail($id);

        // If the setting is a file, delete the file from storage
        if ($setting->type == 'file' && $setting->value) {
            Storage::disk('public')->delete($setting->value);
        }

        $setting->delete();
        return redirect()->route('settings.index', ['group' => $setting->group])->with('success', 'Setting deleted successfully.');
    }

    // Helper function to check if a string is JSON
    protected function isJson($string)
    {
        json_decode($string);
        return json_last_error() === JSON_ERROR_NONE;
    }
}