<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use App\Models\Page;

class PageController extends Controller
{
    // Display a list of pages (include soft deleted)
    public function index()
    {
        $pages = Page::withTrashed()->with('children')->paginate(10);
        return view('dashboard.pages.dynamic.index', compact('pages'));
    }

    // Show form for creating a new page
    public function create()
    {
        $parentPages = Page::whereNull('parent_id')->get();
        return view('dashboard.pages.dynamic.form', compact('parentPages'));
    }

    // Store new page
    public function store(Request $request)
    {
        $data = $this->validatePageData($request);

        // Generate a unique slug if the slug field is empty
        if (empty($data['slug'])) {
            $data['slug'] = $this->generateUniqueSlug($request->title);
        } else {
            // Generate a unique slug from the input
            $data['slug'] = $this->generateUniqueSlug($data['slug']);
        }
        
        // Handle featured image
        if ($request->hasFile('featured_image')) {
            $data['featured_image'] = $request->input('featured_image'); // Save featured image (from media library)
        }

        // Set the user who created the page
        $data['created_by'] = Auth::id();
        
        Page::create($data);

        return redirect()->route('admin.pages.index')->with('success', 'Page created successfully');
    }

    // Show edit form for a specific page
    public function edit(Page $page)
    {
        $parentPages = Page::whereNull('parent_id')->where('id', '!=', $page->id)->get();
        return view('dashboard.pages.dynamic.form', compact('page', 'parentPages'));
    }

    // Update a specific page
    public function update(Request $request, Page $page)
    {
        $data = $this->validatePageData($request);

        // If the user didn't change the slug, generate a unique one from the title
        if (empty($data['slug'])) {
            $data['slug'] = $this->generateUniqueSlug($request->title, $page->id);
        } else {
            // Generate a unique slug based on the manually edited input
            $data['slug'] = $this->generateUniqueSlug($data['slug'], $page->id);
        }

        // Handle featured image update
        if ($request->hasFile('featured_image')) {
            $data['featured_image'] = $request->input('featured_image'); // Update featured image
        }

        // Set the user who updated the page
        $data['updated_by'] = Auth::id();
        
        $page->update($data);

        return redirect()->route('admin.pages.index')->with('success', 'Page updated successfully');
    }

    // Delete a specific page (soft delete)
    public function destroy(Page $page)
    {
        $page->delete();
        return redirect()->route('admin.pages.index')->with('success', 'Page deleted successfully');
    }

    // Restore a trashed page
    public function restore($id)
    {
        $page = Page::withTrashed()->findOrFail($id);
        $page->restore();
        return redirect()->route('admin.pages.index')->with('success', 'Page restored successfully');
    }

    // Permanently delete a page
    public function forceDelete($id)
    {
        $page = Page::withTrashed()->findOrFail($id);
        if ($page->featured_image) {
            Storage::disk('public')->delete($page->featured_image);
        }
        if ($page->gallery_images) {
            foreach ($page->gallery_images as $gallery_image) {
                Storage::disk('public')->delete($gallery_image);
            }
        }
        if ($page->video) {
            Storage::disk('public')->delete($page->video);
        }
        $page->forceDelete();
        return redirect()->route('admin.pages.index')->with('success', 'Page permanently deleted');
    }

    // Display trashed pages (soft deleted)
    public function trashed()
    {
        $trashedPages = Page::onlyTrashed()->paginate(10);
        return view('dashboard.pages.dynamic.trashed', compact('trashedPages'));
    }

    // Show a specific published page in the frontend
    public function show($slug)
    {
        $page = Page::where('slug', $slug)->where('status', 'published')->firstOrFail();
        return view('dashboard.pages.dynamic.show', compact('page'));
    }

    // Helper method for validating form data
    protected function validatePageData($request)
    {
        return $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'required|string|unique:pages,slug,' . ($request->page->id ?? ''),
            'content' => 'nullable',
            'status' => 'required|in:draft,published,scheduled',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:255',
            'featured_image' => 'nullable|image',
            'gallery_images.*' => 'nullable|image',
            'video' => 'nullable|file|mimetypes:video/mp4',
            'parent_id' => 'nullable|exists:pages,id',
            'publish_at' => 'nullable|date',
        ]);
    }

    // Handle gallery uploads
    protected function handleGalleryUpload($galleryImages)
    {
        $gallery = [];
        foreach ($galleryImages as $gallery_image) {
            $gallery[] = $gallery_image->store('pages/galleries', 'public');
        }
        return json_encode($gallery);
    }

    // Helper function to generate a unique slug
    protected function generateUniqueSlug($slug, $id = null)
    {
        // Convert the title or slug to a URL-safe slug
        $slug = Str::slug($slug);

        // Check if the slug exists
        $count = Page::where('slug', 'like', $slug . '%')
                    ->where('id', '!=', $id)
                    ->count();

        // Return the original slug if it's unique
        return $count ? "{$slug}-" . ($count + 1) : $slug;
    }

    // AJAX to check if slug is unique
    public function checkSlug(Request $request)
    {
        $slug = Str::slug($request->slug); // Clean the slug input
        $id = $request->id; // Current page ID (if editing)

        // Check if the slug already exists, excluding the current page (if editing)
        $exists = Page::where('slug', 'like', $slug . '%')
                      ->when($id, function ($query) use ($id) {
                          $query->where('id', '!=', $id);
                      })
                      ->exists();

        // If the slug exists, suggest a unique slug
        if ($exists) {
            $count = Page::where('slug', 'like', $slug . '%')->count();
            $suggestedSlug = $slug . '-' . ($count + 1);

            return response()->json([
                'exists' => true,
                'suggested_slug' => $suggestedSlug
            ]);
        }

        // If slug is available
        return response()->json([
            'exists' => false
        ]);
    }
}