<?php

namespace App\Http\Controllers\Dashboard;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Models\Announcement;
use App\Models\User;

class AnnouncementController extends Controller
{
    // Show a list of all announcements (for admins)
    public function index()
    {
        $announcements = Announcement::all();
        
        return view('dashboard.announcements.index', compact('announcements'));
    }

    // Show the form for creating a new announcement
    public function create()
    {
        // Pass an empty $announcement for the create form
        $announcement = null;
        
        // Optionally, pass any additional data such as users
        $users = User::all(); // Assuming you need users for assigning announcements

        return view('dashboard.announcements.form', compact('announcement', 'users'));
    }

    // Store a new announcement
    public function store(Request $request)
    {
        // Validation for the input fields
        $request->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'type' => 'required|in:normal,information,warning,success',
            'end_date' => 'nullable|date',
            'is_permanent' => 'required|boolean',  // Ensure boolean validation
            'users' => 'nullable|array', // If you're assigning announcements to specific users
        ]);

        // Create a new announcement
        $announcement = Announcement::create([
            'title' => $request->title,
            'message' => $request->message,
            'type' => $request->type,
            'is_permanent' => $request->is_permanent,  // Properly store 0 or 1
            'end_date' => $request->end_date,
        ]);

        // Attach announcement to specific users if provided
        if ($request->has('users')) {
            $announcement->users()->attach($request->users);
        }

        return redirect()->route('announcements.index')->with('success', 'Announcement created successfully.');
    }

    // Show a single announcement
    public function show(Announcement $announcement)
    {
        return view('dashboard.announcements.show', compact('announcement'));
    }

    // Show the form to edit an announcement
    public function edit(Announcement $announcement)
    {
        // Pass the existing announcement for the edit form
        $users = User::all(); // Assuming you want to list users in the form

        return view('dashboard.announcements.form', compact('announcement', 'users'));
    }

    // Update the announcement details
    public function update(Request $request, Announcement $announcement)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'type' => 'required|in:normal,information,warning,success',
            'end_date' => 'nullable|date',
            'is_permanent' => 'required|boolean',  // Ensure boolean validation
            'users' => 'nullable|array', // If you're assigning announcements to specific users
        ]);

        // Update the existing announcement
        $announcement->update([
            'title' => $request->title,
            'message' => $request->message,
            'type' => $request->type,
            'is_permanent' => $request->is_permanent,  // Properly store 0 or 1
            'end_date' => $request->end_date,
        ]);

        // Re-sync users if provided
        if ($request->has('users')) {
            $announcement->users()->sync($request->users);
        }

        return redirect()->route('announcements.index')->with('success', 'Announcement updated successfully.');
    }

    // Delete an announcement
    public function destroy(Announcement $announcement)
    {
        // Detach all user relations before deleting the announcement
        $announcement->users()->detach();

        // Now delete the announcement itself
        $announcement->delete();

        return redirect()->route('announcements.index')->with('success', 'Announcement deleted successfully.');
    }

    // Mark an announcement as seen by the user
    public function markAsSeen(Announcement $announcement)
    {
        $user = Auth::user();

        $user->announcements()->updateExistingPivot($announcement->id, ['is_seen' => true]);

        return response()->json(['status' => 'success']);
    }
}