<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Symfony\Component\Process\Process;
use Symfony\Component\Process\Exception\ProcessFailedException;

class ClearOptimizeCommand extends Command
{
    /*
     * Usage Example
     * 
     * php artisan clear:optimize --skip=composer
     * 
     */ 

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'clear:optimize {--skip= : Comma-separated list of tasks to skip (e.g., composer,cache,route,config,view,optimize,dump)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Clear all caches, run composer update, and optimize the application';

    // Store the results of each task (for web output)
    protected $results = [];

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        // Get the skip argument and convert it to an array
        $skipTasks = $this->option('skip') ? explode(',', $this->option('skip')) : [];

        // Define the tasks with their corresponding skip keywords
        $tasks = [
            'composer'  => ['taskName' => 'Updating Composer dependencies...', 'command' => 'composerUpdate'],
            'cache'     => ['taskName' => 'Clearing application cache...', 'command' => 'cache:clear'],
            'route'     => ['taskName' => 'Clearing route cache...', 'command' => 'route:clear'],
            'config'    => ['taskName' => 'Clearing config cache...', 'command' => 'config:clear'],
            'view'      => ['taskName' => 'Clearing view cache...', 'command' => 'view:clear'],
            'optimize'  => ['taskName' => 'Optimizing the application...', 'command' => 'optimize'],
            'dump'      => ['taskName' => 'Dumping Composer autoload files...', 'command' => 'composerDumpAutoload'],
        ];

        // Display the title and description with bold and styled output
        $this->newLine();
        $this->line('<fg=yellow;options=bold>=== Clear & Optimize Application ===</>');
        $this->newLine();
        $this->line('<fg=white;bg=blue;options=bold>Task: Clear all caches, update dependencies, and optimize the application.</>');
        $this->line('<fg=white;bg=blue;options=bold>Description: This command will run various optimization tasks to improve performance.</>');
        $this->line('<fg=white;bg=blue;options=bold>You can skip tasks if you using --skip argument for example php artisan clear:optimize --skip=composer,dump</>');
        $this->newLine(2);

        if (app()->runningInConsole()) {
            // Running in console: Use progress bar and terminal output
            $progressBar = $this->output->createProgressBar(count($tasks) - count($skipTasks));
            $progressBar->start();
        }

        // Execute each task unless it's in the skip list
        foreach ($tasks as $key => $task) {
            if (in_array($key, $skipTasks)) {
                if (app()->runningInConsole()) {
                    $this->newLine();
                    $this->line("<fg=yellow;options=bold>Skipping: {$task['taskName']}</>");
                } else {
                    $this->results[] = ['task' => $task['taskName'], 'status' => 'Skipped'];
                }
                continue; // Skip this task
            }

            $this->newLine();

            if (app()->runningInConsole()) {
                // Show the task name in console with underline
                $this->newLine();
                $this->line("<fg=yellow;options=bold>{$task['taskName']}</>");
            } else {
                // In web request, add task name to results array
                $this->results[] = ['task' => $task['taskName'], 'status' => 'Running'];
            }

            try {
                // Execute tasks (either a custom function or Artisan command)
                if ($task['command'] === 'composerUpdate') {
                    $this->composerUpdate();
                } elseif ($task['command'] === 'composerDumpAutoload') {
                    $this->composerDumpAutoload();
                } else {
                    $this->callSilent($task['command']);
                }

                // Mark as success
                if (app()->runningInConsole()) {
                    $this->line("<fg=green;options=bold>Success ✔</>");
                } else {
                    $this->results[] = ['task' => $task['taskName'], 'status' => 'Success'];
                }

            } catch (\Exception $e) {
                // If it fails, show "Failed" and log the error
                if (app()->runningInConsole()) {
                    $this->line("<fg=red;options=bold>Failed ✘</>");
                    $this->newLine(1);
                    $this->error("Error in {$task['taskName']}: " . $e->getMessage());
                } else {
                    $this->results[] = ['task' => $task['taskName'], 'status' => 'Failed', 'error' => $e->getMessage()];
                }
            }

            if (app()->runningInConsole()) {
                $this->newLine(1);
                // Advance the progress bar in terminal
                $progressBar->advance();
            }
        }

        if (app()->runningInConsole()) {
            // Finish progress bar in console
            $progressBar->finish();
            $this->newLine(2);
            $this->line('<fg=green;options=bold>Thank you! All tasks have been processed successfully.</>');
        } else {
            // For web request, return the JSON response with the results
            return response()->json([
                'message' => 'Thank you! All tasks have been processed successfully.',
                'results' => $this->results,
            ]);
        }

        return $this->results;
    }

    /**
     * Run Composer update.
     */
    protected function composerUpdate()
    {
        $process = new Process(['composer', 'update']);
        $process->run();

        if (!$process->isSuccessful()) {
            throw new ProcessFailedException($process);
        }
    }

    /**
     * Run Composer dump-autoload.
     */
    protected function composerDumpAutoload()
    {
        $process = new Process(['composer', 'dump-autoload']);
        $process->run();

        if (!$process->isSuccessful()) {
            throw new ProcessFailedException($process);
        }
    }
}